/*
 * USB SQcam WebCam driver based on the ViCam driver written by Joe Burks, 
 * Christopher L Cheney, Pavel Machek, John Tyner, Monroe Williams.
 *
 * Thanks to Theodore Kilgore for his help.
 * Ported to kernel 2.6 by Wilfred Nelson
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * This source code is based heavily on the CPiA webcam driver which was
 * written by Peter Pregler, Scott J. Bertin and Johannes Erdfelt
 *
 * Portions of this code were also copied from usbvideo.c
 *
 * The bayer decoding functions are copied from libgphoto2
 *
 */

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/videodev.h>
#include <linux/usb.h>
#include <linux/vmalloc.h>
#include <linux/slab.h>
#include <linux/proc_fs.h>
#include "usbvideo.h"

// #define SQCAM_DEBUG

#ifndef MODULE_LICENSE
#define MODULE_LICENSE(a)
#endif

#ifndef bool
#define bool int
#endif

#ifdef SQCAM_DEBUG
#define ADBG(lineno,fmt,args...) printk(fmt, jiffies, __FUNCTION__, lineno, ##args)
#define DBG(fmt,args...) ADBG((__LINE__),KERN_DEBUG __FILE__"(%ld):%s (%d):"fmt,##args)
#else
#define DBG(fmn,args...) do {} while(0)
#endif

/* Version Information */
#define DRIVER_VERSION "v1.0"
#define DRIVER_AUTHOR "Marcell Lengyel, miketkf@mailbox.hu"
#define DRIVER_DESC "SQcam WebCam Driver"

/* Define these values to match your device */
#define USB_SQCAM_VENDOR_ID	0x2770
#define USB_SQCAM_PRODUCT_ID	0x9120

#define SQCAM_BYTES_PER_PIXEL 3
#define SQCAM_MAX_READ_SIZE 0x8000
#define SQCAM_MAX_RAW_SIZE 320*240+64
#define SQCAM_MAX_FRAME_SIZE (SQCAM_BYTES_PER_PIXEL*320*240)
#define SQCAM_FRAMES 2
#define SHARPNESS_MIN	0
#define SHARPNESS_MAX	6
#define FRAMERATE_MIN	0
#define FRAMERATE_MAX	6

#define MAX_SQCAM 4

struct sqcam_camera {

	u8 *raw_image;		// raw data captured from the camera
	u8 *framebuf;		// processed data in RGB24 format
         unsigned char gamma_table[256];
	struct video_device vdev;	// v4l video device
	struct usb_device *udev;	// usb device

	struct semaphore busy_lock;	// guard against SMP multithreading

        int channel;
	bool is_initialized;
	bool is_removed ;
	bool is_opened ;
	u8 bulkEndpoint;
        double gamma;
	u32 framebuf_size;	// # of valid bytes in framebuf
	u32 framebuf_read_start;	// position in frame buf that a read is happening at.

};


static unsigned char gamma_table[] = {
0x0, 0x6, 0xa, 0xe, 0x11, 0x13, 0x16, 0x18, 0x1a,
 0x1d, 0x1f, 0x21, 0x22, 0x24, 0x26, 0x28, 0x2a, 0x2b,
  0x2d, 0x2f, 0x30, 0x32, 0x33, 0x35, 0x36, 0x38, 0x39,
0x3b, 0x3c, 0x3e, 0x3f, 0x40, 0x42, 0x43, 0x44, 0x46,
 0x47, 0x48, 0x49, 0x4b, 0x4c, 0x4d, 0x4e, 0x50, 0x51,
0x52, 0x53, 0x54, 0x56, 0x57, 0x58, 0x59, 0x5a, 0x5b,
 0x5c, 0x5e, 0x5f, 0x60, 0x61, 0x62, 0x63, 0x64, 0x65,
 0x66, 0x67, 0x68, 0x69, 0x6a, 0x6b, 0x6d, 0x6e, 0x6f,
   0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77, 0x78, 
0x79, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f, 0x80, 
0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87, 0x88, 0x88,
 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f, 0x90, 0x91,
  0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97, 0x97, 0x98,
   0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9d, 0x9e, 0x9f, 0xa0,
   0xa1, 0xa2, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7, 0xa7,
   0xa8, 0xa9, 0xaa, 0xab, 0xab, 0xac, 0xad, 0xae, 0xaf,
   0xaf, 0xb0, 0xb1, 0xb2, 0xb3, 0xb3, 0xb4, 0xb5, 0xb6,
   0xb6, 0xb7, 0xb8, 0xb9, 0xba, 0xba, 0xbb, 0xbc, 0xbd,
   0xbd, 0xbe, 0xbf, 0xc0, 0xc0, 0xc1, 0xc2, 0xc3, 0xc3,
 0xc4, 0xc5, 0xc6, 0xc6, 0xc7, 0xc8, 0xc9, 0xc9, 0xca,
  0xcb, 0xcc, 0xcc, 0xcd, 0xce, 0xce, 0xcf, 0xd0, 0xd1,
   0xd1, 0xd2, 0xd3, 0xd4, 0xd4, 0xd5, 0xd6, 0xd6, 0xd7, 0xd8,
    0xd9, 0xd9, 0xda, 0xdb, 0xdb, 0xdc, 0xdd, 0xdd, 0xde, 0xdf,
   0xe0, 0xe0, 0xe1, 0xe2, 0xe2, 0xe3, 0xe4, 0xe4, 0xe5, 0xe6,
    0xe6, 0xe7, 0xe8, 0xe9, 0xe9, 0xea, 0xeb, 0xeb, 0xec, 0xed,
   0xed, 0xee, 0xef, 0xef, 0xf0, 0xf1, 0xf1, 0xf2, 0xf3, 0xf3, 0xf4,
   0xf5, 0xf5, 0xf6, 0xf7, 0xf7, 0xf8, 0xf9, 0xf9, 0xfa, 0xfb, 0xfb,
 0xfc, 0xfd, 0xfd, 0xfe};



#define	SQCAM_T(uvd)	((struct sqcam_camera *)((uvd)->user_data))

static int sqcam_probe( struct usb_interface *intf, const struct usb_device_id *id);
static void sqcam_disconnect(struct usb_interface *intf);
static void read_frame(struct sqcam_camera *cam, int framenum);

static int gp_bayer_decode (unsigned char *input, int w, int h,
			     unsigned char *output, int tile);

static void usbvideo_rvfree(void *mem, unsigned long size);
static void *usbvideo_rvmalloc(unsigned long size);
 unsigned long usbvideo_kvirt_to_pa(unsigned long adr);

static int gamma_fill_table     (unsigned char *table, double g);
static int gamma_correct_single (unsigned char *table, unsigned char *data, 
			  unsigned int data_size);

			  
/*static int flags  = FLAGS_DISPLAY_HINTS | FLAGS_OVERLAY_STATS; */
static const int min_canvasWidth  = 8;
static const int min_canvasHeight = 4;


static int
send_control_msg(int set, struct sqcam_camera *cam, u8 request, u16 value,
		 u16 index, unsigned char *cp, u16 size)
{
	int status;

	// for reasons not yet known to me, you can't send USB control messages
	// with data in the module (if you are compiled as a module).  Whatever
	// the reason, copying it to memory allocated as kernel memory then
	// doing the usb control message fixes the problem.

	struct usb_device *udev = cam->udev;
	unsigned char *transfer_buffer = kmalloc(size, GFP_KERNEL);
	    memcpy(transfer_buffer, cp, size);
	/*down(&cam->busy_lock);*/

	status = usb_control_msg(udev,
				 set ? usb_sndctrlpipe(udev, 0) : usb_rcvctrlpipe(udev, 0),
				 request,
				 (set ? USB_DIR_OUT : USB_DIR_IN) | USB_TYPE_VENDOR |
				 USB_RECIP_DEVICE, value, index,
				 transfer_buffer, size, HZ);

	  kfree(transfer_buffer);

	DBG("ctrl msg:: value: %0x, index: %0x, size: %i\n", value, index, size);
	status = min(status, 0);
	if (status < 0) {
		printk(KERN_ERR "send_msg: Failed sending control message, error %d.\n",
		       status);
	printk(KERN_ERR "ctrl msg:: dir: %i, value: %0x, index: %0x, size: %i\n", set, value, index,  size);
	}
	/*up(&cam->busy_lock);*/

	return status;
}

static int
initialize_camera(struct sqcam_camera *cam)
{
	int status;
	unsigned char ch[5], zh;
	
	/*
000001	CS        80 06 00 01 00 00 12 00	

	C<  0000:  1201 1001 ffff ff08 7027 2091 0001 0002  ........p' .....
	    0010:  0001                                     ..

000002	CS        80 06 00 02 00 00 40 00	

	C<  0000:  0902 2700 0101 0080 fa09 0400 0003 ffff  ..'.............
	    0010:  ff00 0705 8102 4000 0007 0502 0240 0000  ......@......@..
	    0020:  0705 8303 0100 03                        .......
	
	
	
	*/
	
	
	zh = 0x00;

	if ((status =
	     send_control_msg(1, cam, 0x0c, 0x06, 0xf0, &zh, 1)) < 0)
		{
		printk(KERN_ERR "initalize: message 1, error %d.\n",status);
		return status;
		}
	zh = 0x00;
	if ((status =
	     send_control_msg(0, cam, 0x0c, 0x07, 0, &zh, 1)) < 0)
		{
		printk(KERN_ERR "initalize: message 2, error %d.\n",status);
		return status;
		}
	
/*

000005	CS        c0 0c 07 00 00 00 01 00	

	C<  0000:  09                                       .

000006	CS        c0 0c 07 00 00 00 01 00	

	C<  0000:  05                                       .

000007	CS        c0 0c 07 00 00 00 01 00	

	C<  0000:  00                                       .

000008	CS        c0 0c 07 00 00 00 01 00	

	C<  0000:  01                                       .

000009	CS        40 0c 06 00 a0 00 01 00	

	C>  0000:  50                                       P

000010	CS        c0 0c 07 00 00 00 01 00	

	C<  0000:  50                                       P


*/	
	
	ch[0] = 0x00;
	if ((status =
	     send_control_msg(0,cam, 0x0c, 0x07, 0, &ch[0], 4)) < 0)
		{
		printk(KERN_ERR "initalize: message 3, error %d.\n",status);
		return status;
		}
	zh = 0x00;
	if ((status =
	     send_control_msg(1, cam, 0x0c, 0x06, 0xa0, &zh, 1)) < 0)
		{
		printk(KERN_ERR "initalize: message 4, error %d.\n",status);
		return status;
		}
	zh = 0x00;
	if ((status =
	     send_control_msg(0, cam, 0x0c, 0x07, 0, &zh, 1)) < 0)
		{
		printk(KERN_ERR "initalize: message, error %d.\n",status);
		return status;
		}
    cam->channel = 1;
	return 0;
}

static int
reset_camera(struct sqcam_camera *cam)
{
	int status;
	unsigned char ch;

	ch = 0xff;
	if ((status =
	     send_control_msg(1, cam, 0x0c, 0xc0, 0x00, &ch, 1)) < 0)
		return status;
	if ((status =
	     send_control_msg(1, cam, 0x0c, 0x06, 0xa0, NULL, 0)) < 0)
		return status;
	if ((status =
	     send_control_msg(0, cam, 0x0c, 0x07, 0, &ch, 1)) < 0)
		return status;
	return 0;
}

static int
sqcam_ioctl(struct inode *inode, struct file *file, unsigned int ioctlnr, unsigned long ul_arg)
{
	void *arg = (void *)ul_arg;
	struct video_device *dev = video_devdata(file);
	struct sqcam_camera *cam = dev->priv;
	int retval = 0;

	if (!cam)
		return -ENODEV;

	/* make this _really_ smp-safe */
	/*if (down_interruptible(&cam->busy_lock))
		return -EINTR;*/

	switch (ioctlnr) {
		/* query capabilites */
	case VIDIOCGCAP:
		{
			struct video_capability b;

			DBG("VIDIOCGCAP\n");
			strcpy(b.name, "SQcam-based Camera");
			b.type = VID_TYPE_CAPTURE;
			b.channels = 1;
			b.audios = 0;
			b.maxwidth = 320;	/* VIDEOSIZE_CIF */
			b.maxheight = 240;
			b.minwidth = 320;	/* VIDEOSIZE_160_120 */
			b.minheight = 240;

			if (copy_to_user(arg, &b, sizeof (b)))
				retval = -EFAULT;

			break;
		}
		/* get/set video source - we are a camera and nothing else */
	case VIDIOCGCHAN:
		{
			struct video_channel v;

			DBG("VIDIOCGCHAN\n");
			if (copy_from_user(&v, arg, sizeof (v))) {
				retval = -EFAULT;
				break;
			}
			if (v.channel != 0 ) {
				retval = -EINVAL;
				break;
			}

			/* v.channel = 0;*/
                        cam->channel = v.channel;
			strcpy(v.name, "Camera");
			v.tuners = 0;
			v.flags = 0;
			v.type = VIDEO_TYPE_CAMERA;
			v.norm = 0;

			if (copy_to_user(arg, &v, sizeof (v)))
				retval = -EFAULT;
			break;
		}

	case VIDIOCSCHAN:
		{
			int v;

			if (copy_from_user(&v, arg, sizeof (v)))
				retval = -EFAULT;
			DBG("VIDIOCSCHAN %d\n", v);

			if (retval == 0 && v != 0)
				retval = -EINVAL;
                         cam->channel = v;
			break;
		}

		/* image properties */
	case VIDIOCGPICT:
		{
			struct video_picture vp;
			DBG("VIDIOCGPICT\n");
			memset(&vp, 0, sizeof (struct video_picture));

			vp.depth = 24;
			vp.palette = VIDEO_PALETTE_RGB24;
			if (copy_to_user
			    (arg, &vp, sizeof (struct video_picture)))
				retval = -EFAULT;
			break;
		}

	case VIDIOCSPICT:
		{
			struct video_picture vp;
			
			if(copy_from_user(&vp, (struct video_picture *) arg,
				sizeof(struct video_picture)))
				retval = -EFAULT;

			else
			{
				DBG("VIDIOCSPICT depth = %d, pal = %d\n", vp.depth,
				    vp.palette);


				if (vp.depth != 24
				    || vp.palette != VIDEO_PALETTE_RGB24)
					retval = -EINVAL;
			}

			break;
		}

		/* get/set capture window */
	case VIDIOCGWIN:
		{
			struct video_window vw;
			vw.x = 0;
			vw.y = 0;
			vw.width = 320;
			vw.height = 240;
			vw.chromakey = 0;
			vw.flags = 0;
			vw.clips = NULL;
			vw.clipcount = 0;

			DBG("VIDIOCGWIN\n");

			if (copy_to_user
			    ((void *) arg, (void *) &vw, sizeof (vw)))
				retval = -EFAULT;

			// I'm not sure what the deal with a capture window is, it is very poorly described
			// in the doc.  So I won't support it now.
			break;
		}

	case VIDIOCSWIN:
		{

			struct video_window *vw = (struct video_window *) arg;
			DBG("VIDIOCSWIN %d x %d\n", vw->width, vw->height);

			if ( vw->width != 320 || vw->height != 240 )
				retval = -EFAULT;
			
			break;
		}

		/* mmap interface */
	case VIDIOCGMBUF:
		{
			struct video_mbuf vm;
			int i;

			DBG("VIDIOCGMBUF\n");
			memset(&vm, 0, sizeof (vm));
			vm.size =
			    SQCAM_MAX_FRAME_SIZE * SQCAM_FRAMES;
			vm.frames = SQCAM_FRAMES;
			for (i = 0; i < SQCAM_FRAMES; i++)
				vm.offsets[i] = SQCAM_MAX_FRAME_SIZE * i;

			if (copy_to_user
			    ((void *) arg, (void *) &vm, sizeof (vm)))
				retval = -EFAULT;

			break;
		}

	case VIDIOCMCAPTURE:
		{
			struct video_mmap vm;
			// int video_size;

			if (copy_from_user
			    ((void *) &vm, (void *) arg, sizeof (vm))) {
				retval = -EFAULT;
				break;
			}

			DBG("VIDIOCMCAPTURE frame=%d, height=%d, width=%d, format=%d.\n",vm.frame,vm.width,vm.height,vm.format);

			if ( vm.frame >= SQCAM_FRAMES || vm.format != VIDEO_PALETTE_RGB24 )
				retval = -EINVAL;

			// in theory right here we'd start the image capturing
			// (fill in a bulk urb and submit it asynchronously)
			//
			// Instead we're going to do a total hack job for now and
			// retrieve the frame in VIDIOCSYNC

			break;
		}

	case VIDIOCSYNC:
		{
			int frame;

			if (copy_from_user((void *) &frame, arg, sizeof (int))) {
				retval = -EFAULT;
				break;
			}
			DBG("VIDIOCSYNC: %d\n", frame);
			read_frame(cam, frame);

			break;
		}

		/* pointless to implement overlay with this camera */
	case VIDIOCCAPTURE:
	case VIDIOCGFBUF:
	case VIDIOCSFBUF:
	case VIDIOCKEY:
		retval = -EINVAL;
		break;

		/* tuner interface - we have none */
	case VIDIOCGTUNER:
	case VIDIOCSTUNER:
	case VIDIOCGFREQ:
	case VIDIOCSFREQ:
		retval = -EINVAL;
		break;

		/* audio interface - we have none */
	case VIDIOCGAUDIO:
	case VIDIOCSAUDIO:
		retval = -EINVAL;
		break;
	default:
		retval = -ENOIOCTLCMD;
		break;
	}

	/*up(&cam->busy_lock);*/
	return retval;
}

static int
sqcam_open(struct inode *inode, struct file *file)
{
    	struct video_device *dev = video_devdata(file);
	struct sqcam_camera *cam = 
	    (struct sqcam_camera *) dev->priv;
	unsigned char siz;
	int n;
	DBG("open\n");
	
	if (!cam) {
		printk(KERN_ERR
		       "sqcam video_device improperly initialized");
	}

	int intr = down_interruptible(&cam->busy_lock);
	if (intr)
		return -EINTR;

	if (cam->is_opened) {
		printk(KERN_INFO
		       "sqcam_open called on already opened camera");
		up(&cam->busy_lock);
		return -EBUSY;
	}

	cam->raw_image = kmalloc(SQCAM_MAX_RAW_SIZE, GFP_KERNEL);
	if (!cam->raw_image) {
		if (!cam->raw_image) {
			up(&cam->busy_lock);
			return -ENOMEM;
		}
	}

		cam->framebuf =
		    usbvideo_rvmalloc(SQCAM_MAX_FRAME_SIZE * SQCAM_FRAMES);
	if (!cam->framebuf) {
		if (!cam->framebuf) {
			kfree(cam->raw_image);
			up(&cam->busy_lock);
			return -ENOMEM;
		}
	}

	if (!cam->is_initialized) {
		n = initialize_camera(cam);
            if (n >= 0)
		cam->is_initialized = 1;
	}
	siz = 0x00;
	if ((n = send_control_msg(1, cam, 0x0c, 0xc0, 0, &siz, 0)) < 0)
		{
		printk(KERN_ERR "sqcam_open: message 1, error %d.\n",n);
		return n;
		}
	siz = 0x61; // 0x60 -> 160x120, 0x61 -> 320x240
	if (n >= 0) n = send_control_msg(1, cam, 0x0c, 0x06, siz, NULL, 0);
		if (n < 0)
		{
		printk(KERN_ERR "sqcam_open: message 2, error %d.\n",n);
		return n;
		}
	siz = 0x00;
	if (n >= 0) n = send_control_msg(0, cam, 0x0c, 0x07, 0x00, &siz, 1);
		if (n < 0)
		{
		printk(KERN_ERR "sqcam_open: message 3, error %d.\n",n);
		return n;
		}
        if (n < 0) {
         cam->is_initialized = 0;
         return -EIO;
        }
	cam->is_opened = 1;

	up(&cam->busy_lock);

	return 0;
}

static int 
sqcam_close(struct inode *inode, struct file *file)
{
	struct video_device *dev = video_devdata(file);
	struct sqcam_camera *cam = dev->priv;
	struct usb_device *udev;
	DBG("close\n");


	/* it's not the end of the world if
	 * we fail to turn the camera off.
	 */


	kfree(cam->raw_image);
	usbvideo_rvfree(cam->framebuf, SQCAM_MAX_FRAME_SIZE * SQCAM_FRAMES);

	down(&cam->busy_lock);

	cam->is_opened = 0;
	reset_camera(cam);
   

	udev = cam->udev;

	up(&cam->busy_lock);

	if (!cam->is_opened && !udev) {
		kfree(cam);
	}

   return 0;
}


#define DATA_HEADER_SIZE 64

void sqcam_decode_bayer(const u8 *data, u8 *rgb, struct sqcam_camera *cam)
{
	int j;
	unsigned char b;
	unsigned char *src;
	unsigned char *dst;

	src = (u8 *) data + DATA_HEADER_SIZE;
	dst = rgb;

        /* Reverse the data, else the pic is upside down. */
        for (j = 0; j < 320*240 / 2; j++) {
	      b = src[j];
	      src[j] = src[320*240 - 1 - j];
	      src[320*240 - 1 - j] = b;
        } 
	gp_bayer_decode(src, 320, 240, dst, cam->channel);
       /*gamma_correct_single (gamma_table, dst  , 320 * 240);*/

}

static void
read_frame(struct sqcam_camera *cam, int framenum)
{
	unsigned char siz, msg;
	int n, read;
	int actual_length;


	if (!cam->udev) {
		return;
	}
	msg = 0x03; 
	        read = 320 * 240 + 64;
	// read 'read' bytes in SQCAM_MAX_READ_SIZE chunks

		
	do {
		siz = 0x50;
		n = send_control_msg(1, cam, 0x0c,0x03, 
				(read > SQCAM_MAX_READ_SIZE) ? SQCAM_MAX_READ_SIZE : read, 
				&siz, 1);
		if (n < 0) {
			printk(KERN_ERR
		       		" sqcam_read_frame: Problem sending frame capture control message");
			goto done;
		}
		n = usb_bulk_msg(cam->udev,
			 usb_rcvbulkpipe(cam->udev, cam->bulkEndpoint),
			 cam->raw_image + (320*240+64 - read),
			 (read > SQCAM_MAX_READ_SIZE) ? SQCAM_MAX_READ_SIZE : read,
			 &actual_length, 500);
		read -= actual_length;
		DBG("blk msg:: size: %d\n", actual_length);
		if (n < 0) {
			printk(KERN_ERR "Problem during bulk read of frame data: %d\n", n);
			read = 0;
		}
	} while (read);	
	
	sqcam_decode_bayer(cam->raw_image,
			 cam->framebuf +
			 framenum * SQCAM_MAX_FRAME_SIZE,
                         cam );
	cam->framebuf_size =
	    320 * 240 * SQCAM_BYTES_PER_PIXEL;
	cam->framebuf_read_start = 0;
 done: 
  DBG("sqcam_read_frame: exiting \n");
}

static ssize_t
sqcam_read( struct file *file, char *buf, size_t count, loff_t *ppos )
{
	struct video_device *dev = video_devdata(file);
	struct sqcam_camera *cam = dev->priv;
	
	int intr = down_interruptible(&cam->busy_lock);
	if (intr)
		return -EINTR;


	DBG("read %d bytes.\n", (int) count);

	if (*ppos >= SQCAM_MAX_FRAME_SIZE) {
		*ppos = 0;
		return 0;
	}

	if (*ppos == 0) {
		read_frame(cam, 0);
	}

	count = min_t(size_t, count, SQCAM_MAX_FRAME_SIZE - *ppos);

	if (copy_to_user(buf, &cam->framebuf[*ppos], count)) {
		count = -EFAULT;
	} else {
		*ppos += count;
	}

	if (count == SQCAM_MAX_FRAME_SIZE) {
		*ppos = 0;
	}
	DBG("read %d bytes. finished\n", (int) count);

	up(&cam->busy_lock);
	return count;
}






static int
sqcam_mmap(struct file *file, struct vm_area_struct *vma)
{
	// TODO: allocate the raw frame buffer if necessary
	unsigned long start = vma->vm_start;
	unsigned long size  = vma->vm_end-vma->vm_start;
	unsigned long page, pos;
	struct video_device *dev = video_devdata(file);

	DBG("sqcam_mmap: %ld\n", size);

	/* We let mmap allocate as much as it wants because Linux was adding 2048 bytes
	 * to the size the application requested for mmap and it was screwing apps up.
	 if (size > SQCAM_FRAMES*SQCAM_MAX_FRAME_SIZE)
	 return -EINVAL;
	 */
	
	struct sqcam_camera *cam = dev->priv;
	if (!cam)
		return -ENODEV;

        
	if (down_interruptible(&cam->busy_lock))
		return -EINTR;


	if (!cam->framebuf) {
		cam->framebuf =
		    usbvideo_rvmalloc(SQCAM_MAX_FRAME_SIZE * SQCAM_FRAMES);
		if (!cam->framebuf) {
			up(&cam->busy_lock);
			return -ENOMEM;
		}
	}

	pos = (unsigned long) (cam->framebuf);
	while (size > 0) {
		page = usbvideo_kvirt_to_pa(pos);
		if (remap_page_range(vma, start, page, PAGE_SIZE, PAGE_SHARED)) {
			up(&cam->busy_lock);
			return -EAGAIN;
		}
		start += PAGE_SIZE;
		pos += PAGE_SIZE;
		if (size > PAGE_SIZE)
			size -= PAGE_SIZE;
		else
			size = 0;
	}

	up(&cam->busy_lock);

	return 0;
}




/* table of devices that work with this driver */
static struct usb_device_id sqcam_table[] = {
	{USB_DEVICE(USB_SQCAM_VENDOR_ID, USB_SQCAM_PRODUCT_ID)},
	{}			/* Terminating entry */
};

MODULE_DEVICE_TABLE(usb, sqcam_table);

static struct file_operations sqcam_fops = {
	.owner =  THIS_MODULE,
	.open =   sqcam_open,
	.release = sqcam_close,
	.read =   sqcam_read,
	.mmap =  sqcam_mmap,
	.ioctl = sqcam_ioctl,
	.llseek = no_llseek,
};

static struct usb_driver sqcam_driver = {
        .owner          = THIS_MODULE,
        .name           = "sqcam",
        .probe          = sqcam_probe,
        .disconnect     = sqcam_disconnect,
        .id_table       = sqcam_table,
};



static void sqcam_exclusive_release(struct video_device *vdev)
{
        struct sqcam_camera *cam = vdev->priv;
/*FIXME: this could be called after sqcam_close or sqcam_disconnect and print an error message */	
	if (!cam){
	 printk(KERN_ERR"sqcam_release: invalid data");
                return ;
	}
	 printk(KERN_ERR"sqcam_release: video unregister\n");
	video_unregister_device(&cam->vdev);
	 printk(KERN_ERR"sqcam_release: video unregister returned\n");
       down(&cam->busy_lock);

	if (cam->is_opened) {
		cam->is_removed = 1;
	} 
	 printk(KERN_ERR"sqcam_release: usb driver release\n");
       usb_driver_release_interface(&sqcam_driver,
                                     cam->udev->actconfig->interface[0]);
	 printk(KERN_ERR"sqcam_release: usb driver release returned\n");
	up(&cam->busy_lock);

	
	vdev->users--;
	return ;
}




static struct video_device sqcam_template = {
	.owner = THIS_MODULE,
	.name  = "SQcam-based USB Camera",
	.type   = VID_TYPE_CAPTURE,
	.hardware = VID_HARDWARE_SE401, /* need an own value */
	.release = &sqcam_exclusive_release,
        .fops  =  &sqcam_fops,
	.minor = -1,
};

/**
 *	sqcam_probe
 *
 *	Called by the usb core when a new device is connected that it thinks
 *	this driver might be interested in.
 */
static int
sqcam_probe(struct usb_interface *intf,
		  const struct usb_device_id *id)
{
	int nas, bulkEndpoint = 0;
        struct usb_device *dev = interface_to_usbdev(intf);
	const struct usb_host_interface *interface;
	const struct usb_endpoint_descriptor *endpoint;
	struct sqcam_camera *cam;
	__u8 ifnum = intf->altsetting->desc.bInterfaceNumber;

	/* See if the device offered us matches what we can accept */
	if ((dev->descriptor.idVendor != USB_SQCAM_VENDOR_ID) ||
	    (dev->descriptor.idProduct != USB_SQCAM_PRODUCT_ID)) {
		return -ENODEV;
	}

	printk(KERN_INFO "SQcam based webcam connected\n");

	nas = dev->actconfig->interface[ifnum]->num_altsetting;
	if (nas != 1) {
		printk(KERN_WARNING
		       "Expected only one alternate setting for this camera!\n");
		return -EINVAL;
	}

	interface = &dev->actconfig->interface[ifnum]->altsetting[0];
	DBG(KERN_DEBUG "Interface %d. has %u. endpoints!\n",
	       interface->desc.bInterfaceNumber, (unsigned) (interface->desc.bNumEndpoints));
	endpoint = &interface->endpoint[0].desc;

	if ((endpoint->bEndpointAddress & 0x80) &&
	    ((endpoint->bmAttributes & 3) == 0x02)) {
		/* we found a bulk in endpoint */
		bulkEndpoint = endpoint->bEndpointAddress;
	} else {
		printk(KERN_ERR
		       "No bulk in endpoint was found ?! (this is bad)\n");
	}

	if ((cam =
	     kmalloc(sizeof (struct sqcam_camera), GFP_KERNEL)) == NULL) {
		printk(KERN_WARNING
		       "could not allocate kernel memory for sqcam_camera struct\n");
		return -ENOMEM;
	}

	memset(cam, 0, sizeof (struct sqcam_camera));

	init_MUTEX(&cam->busy_lock);

	memcpy(&cam->vdev, &sqcam_template,
	       sizeof (sqcam_template));
	cam->vdev.priv = cam;	// sort of a reverse mapping for those functions that get vdev only

	cam->udev = dev;
	cam->bulkEndpoint = bulkEndpoint;
	cam->is_initialized = 0;
	cam->is_opened = 0;
	cam->gamma = 0.65;
        gamma_fill_table (gamma_table, cam->gamma);
	
	if (video_register_device(&cam->vdev, VFL_TYPE_GRABBER, -1) == -1) {
		kfree(cam);
		printk(KERN_WARNING "video_register_device failed\n");
		return -EIO;
	}

	printk(KERN_INFO "SQcam webcam driver v0.1b now controlling video device %d\n",cam->vdev.minor);
	usb_set_intfdata (intf, cam);


	return 0;
}

static void
sqcam_disconnect(struct usb_interface *intf)
{
    struct sqcam_camera *cam = usb_get_intfdata(intf);


        if (!cam)
                return;
	reset_camera(cam);
        usb_set_intfdata(intf, NULL);
	video_unregister_device(&cam->vdev);
       down(&cam->busy_lock);

	cam->udev = NULL;


	/* the only thing left to do is synchronize with
	 * our close/release function on who should release
	 * the camera memory. if there are any users using the
	 * camera, it's their job. if there are no users,
	 * it's ours.
	 */

	up(&cam->busy_lock);

	if (!cam->is_opened) {
		kfree(cam);
	}
	printk(KERN_DEBUG "SQCam-based WebCam disconnected\n");
	
}

static int __init
usb_sqcam_init(void)
{
	int retval;
	DBG(KERN_INFO "SQcam-based WebCam driver startup\n");

	retval = usb_register(&sqcam_driver);
	if (retval)
		printk(KERN_WARNING "usb_register failed!\n");
	return retval;
}

static void __exit
usb_sqcam_exit(void)
{
	DBG(KERN_INFO
	       "SQcam-based WebCam driver shutdown\n");

	usb_deregister(&sqcam_driver);
}

module_init(usb_sqcam_init);
module_exit(usb_sqcam_exit);

MODULE_AUTHOR(DRIVER_AUTHOR);
MODULE_DESCRIPTION(DRIVER_DESC);
MODULE_LICENSE("GPL");
/* bayer.c
 *
 * Copyright  2001 Lutz Mller <urc8@rz.uni-karlsruhe.de>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, 
 * but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details. 
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/* This is from libgphoto2 */


static int tile_colors[8][4] = {
	{0, 1, 1, 2},
	{1, 0, 2, 1},
	{2, 1, 1, 0},
	{1, 2, 0, 1},
	{0, 1, 1, 2},
	{1, 0, 2, 1},
	{2, 1, 1, 0},
	{1, 2, 0, 1}};

#define RED 0
#define GREEN 1
#define BLUE 2

static int
gp_bayer_expand (unsigned char *input, int w, int h, unsigned char *output,
		 int tile)
{
	int x, y, i;
	int colour, bayer;
	char *ptr = input;

	switch (tile) {

		case 0:
		case 1:
		case 2:
		case 3: 

			for (y = 0; y < h; ++y)
				for (x = 0; x < w; ++x, ++ptr) {
					bayer = (x&1?0:1) + (y&1?0:2);

					colour = tile_colors[tile][bayer];

					i = (y * w + x) * 3;

					output[i+RED]    = 0;
					output[i+GREEN]  = 0;
					output[i+BLUE]   = 0;
					output[i+colour] = *ptr;
				}
			break;

		case 4: //BAYER_TILE_RGGB_INTERLACED:
		case 5: //BAYER_TILE_GRBG_INTERLACED:
		case 6: //BAYER_TILE_BGGR_INTERLACED:
		case 7: //BAYER_TILE_GBRG_INTERLACED:
 

			for (y = 0; y < h; ++y, ptr+=w)
				for (x = 0; x < w; ++x) {
					bayer = (x&1?0:1) + (y&1?0:2);
	
					colour = tile_colors[tile][bayer];
	
					i = (y * w + x) * 3;

					output[i+RED]    = 0;
					output[i+GREEN]  = 0;
					output[i+BLUE]   = 0;
					output[i+colour] = (x&1)? ptr[x>>1]:ptr[(w>>1)+(x>>1)];
				}
			break;
	}

	return (0);
}

#define AD(x, y, w) ((y)*(w)*3+3*(x))

static int
gp_bayer_interpolate (unsigned char *image, int w, int h, int tile)
{
	int x, y, bayer;
	int p0, p1, p2, p3;
	int div, value;


	switch (tile) {
	default:
	case 0: //BAYER_TILE_RGGB:
	case 4: //BAYER_TILE_RGGB_INTERLACED:
		p0 = 0; p1 = 1; p2 = 2; p3 = 3;
		break;
	case 1: //BAYER_TILE_GRBG:
	case 5: //BAYER_TILE_GRBG_INTERLACED:
		p0 = 1; p1 = 0; p2 = 3; p3 = 2;
		break;
	case 2: //BAYER_TILE_BGGR:
	case 6: //BAYER_TILE_BGGR_INTERLACED:
		p0 = 3; p1 = 2; p2 = 1; p3 = 0;
		break;
	case 3: //BAYER_TILE_GBRG:
	case 7: //BAYER_TILE_GBRG_INTERLACED:	
		p0 = 2; p1 = 3; p2 = 0; p3 = 1;
		break;
	}


	//p0 = 3; p1 = 2; p2 = 1; p3 = 0; // 2
	//p0 = 0; p1 = 1; p2 = 2; p3 = 3;


	for (y = 0; y < h; y++)
		for (x = 0; x < w; x++) {
			bayer = (x&1?0:1) + (y&1?0:2);
			if ( bayer == p0 ) {
				
				/* red. green lrtb, blue diagonals */
				div = value = 0;
				if (y) {
					value += image[AD(x,y-1,w)+GREEN];
					div++;
				}
				if (y < (h - 1)) {
					value += image[AD(x,y+1,w)+GREEN];
					div++;
				}
				if (x) {
					value += image[AD(x-1,y,w)+GREEN];
					div++;
				}
				if (x < (w - 1)) {
					value += image[AD(x+1,y,w)+GREEN];
					div++;
				}
				image[AD(x,y,w)+GREEN] = value / div;

				div = value = 0;
				if ((y < (h - 1)) && (x < (w - 1))) {
					value += image[AD(x+1,y+1,w)+BLUE];
					div++;
				}
				if ((y) && (x)) {
					value += image[AD(x-1,y-1,w)+BLUE];
					div++;
				}
				if ((y < (h - 1)) && (x)) {
					value += image[AD(x-1,y+1,w)+BLUE];
					div++;
				}
				if ((y) && (x < (w - 1))) {
					value += image[AD(x+1,y-1,w)+BLUE];
					div++;
				}
				image[AD(x,y,w)+BLUE] = value / div;

			} else if (bayer == p1) {
				
				/* green. red lr, blue tb */
				div = value = 0;
				if (x < (w - 1)) {
					value += image[AD(x+1,y,w)+RED];
					div++;
				}
				if (x) {
					value += image[AD(x-1,y,w)+RED];
					div++;
				}
				image[AD(x,y,w)+RED] = value / div;

				div = value = 0;
				if (y < (h - 1)) {
					value += image[AD(x,y+1,w)+BLUE];
					div++;
				}
				if (y) {
					value += image[AD(x,y-1,w)+BLUE];
					div++;
				}
				image[AD(x,y,w)+BLUE] = value / div;

			} else if ( bayer == p2 ) {
				
				/* green. blue lr, red tb */
				div = value = 0;
				
				if (x < (w - 1)) {
					value += image[AD(x+1,y,w)+BLUE];
					div++;
				}
				if (x) {
					value += image[AD(x-1,y,w)+BLUE];
					div++;
				}
				image[AD(x,y,w)+BLUE] = value / div;

				div = value = 0;
				if (y < (h - 1)) {
					value += image[AD(x,y+1,w)+RED];
					div++;
				}
				if (y) {
					value += image[AD(x,y-1,w)+RED];
					div++;
				}
				image[AD(x,y,w)+RED] = value / div;

			} else {
				
				/* blue. green lrtb, red diagonals */
				div = value = 0;

				if (y) {
					value += image[AD(x,y-1,w)+GREEN];
					div++;
				}
				if (y < (h - 1)) {
					value += image[AD(x,y+1,w)+GREEN];
					div++;
				}
				if (x) {
					value += image[AD(x-1,y,w)+GREEN];
					div++;
				}
				if (x < (w - 1)) {
					value += image[AD(x+1,y,w)+GREEN];
					div++;
				}
				image[AD(x,y,w)+GREEN] = value / div;

				div = value = 0;
				if ((y < (h - 1)) && (x < (w - 1))) {
					value += image[AD(x+1,y+1,w)+RED];
					div++;
				}
				if ((y) && (x)) {
					value += image[AD(x-1,y-1,w)+RED];
					div++;
				}
				if ((y < (h - 1)) && (x)) {
					value += image[AD(x-1,y+1,w)+RED];
					div++;
				}
				if ((y) && (x < (w - 1))) {
					value += image[AD(x+1,y-1,w)+RED];
					div++;
				}
				image[AD(x,y,w)+RED] = value / div;
			}
		}

	return (0);
}

static int
gp_bayer_decode (unsigned char *input, int w, int h, unsigned char *output,
		 int tile)
{
	gp_bayer_expand (input, w, h, output, tile);
	gp_bayer_interpolate (output, w, h, tile);

	return (0);
}

static void *usbvideo_rvmalloc(unsigned long size)
{
	void *mem;
	unsigned long adr;

	size = PAGE_ALIGN(size);
	mem = vmalloc_32(size);
	if (!mem)
		return NULL;

	memset(mem, 0, size); /* Clear the ram out, no junk to the user */
	adr = (unsigned long) mem;
	while (size > 0) {
		SetPageReserved(vmalloc_to_page((void *)adr));
		adr += PAGE_SIZE;
		size -= PAGE_SIZE;
	}

	return mem;
}

static void usbvideo_rvfree(void *mem, unsigned long size)
{
	unsigned long adr;

	if (!mem)
		return;

	adr = (unsigned long) mem;
	while ((long) size > 0) {
		ClearPageReserved(vmalloc_to_page((void *)adr));
		adr += PAGE_SIZE;
		size -= PAGE_SIZE;
	}
	vfree(mem);
}

 unsigned long usbvideo_kvirt_to_pa(unsigned long adr)
{
	unsigned long kva, ret;

	kva = (unsigned long) page_address(vmalloc_to_page((void *)adr));
	kva |= adr & (PAGE_SIZE-1); /* restore the offset */
	ret = __pa(kva);
	return ret;
}


static int
gamma_correct_triple (unsigned char *table_red,
		      unsigned char *table_green,
		      unsigned char *table_blue,
		      unsigned char *data, unsigned int size)
{
	int x;

	for (x = 0; x < (size * 3); x += 3) {
		data[x + 0] = table_red  [data[x + 0]];
		data[x + 1] = table_green[data[x + 1]];
		data[x + 2] = table_blue [data[x + 2]];
	}

	return (0);
}

static int
gamma_correct_single (unsigned char *table, unsigned char *data,
			 unsigned int size)
{
	return (gamma_correct_triple (table, table, table, data, size));
}

  
  


static int
gamma_fill_table (unsigned char *table, double g)
{
/*	int x;

	for (x = 0; x < 256; x++)

		table[x] = 255 * pow ((double) x/255., g);
*/
	return (0);
}

